#!/usr/local/bin/perl
# 
# oc4jctl_lib.pm
# 
# Copyright (c) 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      oc4jctl_lib.pm - OC4J Control Action Script Generic Library Module
#
#    DESCRIPTION
#      oc4jctl_lib.pm - Generic library module for OC4J Control
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    dsemler     10/04/12 - fix comparison operator in stop loop
#    dsemler     10/03/12 - Bug 14619154 : set the isleep_interval default to
#                           1 second
#    ekhabie     05/16/12 - Update to new ShutdownOc4j location
#    jgrout      03/09/12 - Fix bug 13831825
#    jgrout      02/07/12 - Created

package oc4jctl_lib;

use strict;
use warnings;
use LWP;
use Sys::Hostname;
use File::Spec::Functions;
use Config;
use oc4jctl_common;
use s_oc4jctl_lib;

our $VERSION = '1';

use English;
use Exporter;

# Implementation constants
use constant {
  SHUTTER_CLASS => "oracle.cluster.consoleapps.shutter.ShutdownOc4j"
};

# Do NOT use constants as hash keys - Perl 5.8 and later don't support it

# USR_ORA_ENV keys

our $config_dir_key = "JAZN_DIR";
our $dbwlm_logging_level_key = "DBWLM_LOGGING_LEVEL";
our $http_port_key = "HTTP_PORT";
our $isleep_interval_key = "ISLEEP_INTERVAL";
our $min_heap_size_key = "MIN_HEAP_SIZE";
our $min_perm_size_key = "MIN_PERM_SIZE";
our $max_heap_size_key = "MAX_HEAP_SIZE";
our $max_perm_size_key = "MAX_PERM_SIZE";
our $script_debug_key = "SCRIPT_DEBUG";
our $srvm_trace_level_key = "SRVM_TRACE_LEVEL";
our $use_scan_ip_key = "use_scan_IP";

# Global variable keys

our $j2ee_home_key = "J2EE_HOME";
our $jlib_key = "JLIB";
our $jre_key = "JRE";
our $pid_file_key = "PID_FILE";

our @exp_vars;
our $oc4j_pid_list_match_regex;
our $oc4j_shutter_pid_list_match_regex;

our %envhash;
our %globhash;

our (@ISA, @EXPORT);

# Export in a BEGIN block to avoid compilation failure

BEGIN {
  require Exporter;
  @ISA = qw(Exporter);

  my @exp_const = qw(SUCC_CODE FAIL_CODE ERROR_PID NULL_PID
                     STATE_ONLINE STATE_OFFLINE STATE_UNKNOWN STATE_FAILED
                    );

  my @exp_func = qw(init_globals oc4j_start oc4j_stop oc4j_check oc4j_clean
                   );

  my @exp = (@exp_const, @exp_func);
  @EXPORT  = @exp;
}

sub init_globals
#---------------------------------------------------------------------
# Function: Initialize globals
#
# Args    : None
#
# Returns : SUCC_CODE (operation succeeded)
#           FAIL_CODE (operation failed)
#---------------------------------------------------------------------
{

  my $oracle_home = $ENV{"ORACLE_HOME"};
  my $jlib = catfile($oracle_home, "jlib");
  my $j2ee_home = catfile($oracle_home, "oc4j", "j2ee", "home");

  my $config_dir = catfile($j2ee_home, "OC4J_DBWLM_config");
  my $pid_file = catfile($config_dir, ".oc4jpid_" . hostname);

  # Initialize environment hash
  %envhash = ($config_dir_key => $config_dir ,
              $dbwlm_logging_level_key => "INFO" ,
              $http_port_key => 8888 ,
              $isleep_interval_key => 1 ,
              $min_heap_size_key => "128M" ,
              $min_perm_size_key => undef ,
              $max_heap_size_key => "384M" ,
              $max_perm_size_key => undef ,
              $script_debug_key => undef ,
              $srvm_trace_level_key => undef,
              $use_scan_ip_key => undef);

  # Initialize global hash
  %globhash = ($j2ee_home_key => $j2ee_home ,
               $jlib_key => $jlib ,
               $jre_key => get_jre() ,
               $pid_file_key => $pid_file);

  # Parse USR_ORA_ENV into environment hash
  my @envpairlist = split(/\s+/, $ENV{"_USR_ORA_ENV"});
  foreach my $envpair (@envpairlist)
  {
    my @envlist = split(/\=/, $envpair);
    if (@envlist == 2)
    {
      my ($opt, $optval) = @envlist;
      if (exists $envhash{$opt})
      {
        $envhash{$opt} = $optval;
        next;
      }
    }
    print STDERR
      " CRS_ERROR:WARNING: Invalid data $envpair in _USR_ORA_ENV\n";
  }

  if (defined($envhash{$script_debug_key}))
  {
    set_debug_out();
  }
  else
  {
    unset_debug_out();
  }

  return s_init_globals();

}

sub get_jre
#---------------------------------------------------------------------
# Function: Return JRE file path
#
# Args    : None
#
# Returns : JRE file path
#---------------------------------------------------------------------
{
  return s_get_jre();
}

sub get_jstack
#---------------------------------------------------------------------
# Function: Return jstack command file path
#
# Args    : None
#
# Returns : jstack command file path
#---------------------------------------------------------------------
{
  return s_get_jstack();
}

sub get_pid_hash
#---------------------------------------------------------------------
# Function: Return a hash of Java PIDs to command lines
#
# Args    : None
#
# Returns : PID hash (key PID, value corresponding command line)
#---------------------------------------------------------------------
{
  return s_get_pid_hash();
}

sub get_matching_pid_list
#---------------------------------------------------------------------
# Function: Get a PID list matching a regex
#
# Args    : Compiled match regex
#
# Returns : List of matching PIDs (if any)
#---------------------------------------------------------------------
{
  my ($match_regex) = @_;

  my @pid_list;
  my %pid_hash = get_pid_hash();

  foreach my $pid (sort keys(%pid_hash))
  {
    my $command = $pid_hash{$pid};
    if ($command =~ $match_regex)
    {
      debug_out("Match for PID $pid");
      push @pid_list, $pid;
    }
  }
  return @pid_list;
}

sub async_start
#---------------------------------------------------------------------
# Function: Start command asynchronously in the background
#
# Args    : List with program name and its arguments
#
# Returns : Result of system command
#---------------------------------------------------------------------
{
  return s_async_start(@_);
}

sub get_oc4j_pid_list
#---------------------------------------------------------------------
# Function: Get a list of OC4J container PID(s).
#           Match against OC4J JAR name and OC4J config file name
#
# Args    : None
#
# Returns : List of OC4J container PID(s) (if any)
#---------------------------------------------------------------------
{
  debug_out("Get OC4J PIDs");

  # Compile the match regex on first entry
  if (!$oc4j_pid_list_match_regex)
  {
    my $j2ee_home = $globhash{$j2ee_home_key};
    my $oc4j_jar = catfile($j2ee_home, "oc4j.jar");
    my $oc4j_jar_qm = quotemeta($oc4j_jar);
    my $config_qm = quotemeta("-config");
    my $config_dir = $envhash{$config_dir_key};
    my $server_xml_qm = quotemeta(catfile($config_dir, "server.xml"));
    $oc4j_pid_list_match_regex =
      qr/$oc4j_jar_qm.*$config_qm.*$server_xml_qm/p;
  }

  # Return matching PID(s) to caller
  my @oc4j_pid_list =
    get_matching_pid_list($oc4j_pid_list_match_regex);
  debug_out("Done Getting OC4J PIDs");
  return @oc4j_pid_list;
}

sub get_oc4j_shutter_pid_list
#---------------------------------------------------------------------
# Function: Get a list of OC4J shutter PID(s).
#           Match against ADMIN JAR name, shutter class and port number
#
# Args    : None
#
# Returns : List of OC4J shutter command PID(s) (if any)
#---------------------------------------------------------------------
{
  debug_out("Get OC4J shutter PIDs");

  # Compile the match regex on first entry
  if (!$oc4j_shutter_pid_list_match_regex)
  {
    my $shutter_qm = quotemeta(SHUTTER_CLASS);
    my $ormi_qm = quotemeta("ormi://");
    my $port_qm = quotemeta($ENV{"OC4J_PORT"});
    $oc4j_shutter_pid_list_match_regex =
      qr/($shutter_qm)\s+$ormi_qm\S+:($port_qm)/;
  }

  # Return matching PID(s) to caller
  return get_matching_pid_list($oc4j_shutter_pid_list_match_regex);
}

sub read_pid_from_oc4j_pid_file
#---------------------------------------------------------------------
# Function: Read the OC4J PID from the OC4J PID file
#
# Args    : Pid file
#
# Returns : OC4J Container PID (if PID file is found)
#           NULL_PID (if PID file is not found)
#---------------------------------------------------------------------
{
  my ($pid_file) = @_;
  my $oc4j_pid = NULL_PID;

  if (-e $pid_file && -r $pid_file)
  {
    open(PF, $pid_file);
    while(<PF>)
    {
      chomp;
      $oc4j_pid = $_;
    }
    close(PF);
  }
  return $oc4j_pid;
}

sub write_pid_to_oc4j_pidfile_and_check
#---------------------------------------------------------------------
# Function: Write the OC4J PID to the OC4J PID file
#           and check that it was created successfully.
#
# Args    : PID file
#           OC4J PID to write
#
# Returns : SUCC_CODE (operation succeeded)
#           FAIL_CODE (operation failed)
#---------------------------------------------------------------------
{
  my ($pid_file, $oc4j_pid) = @_;

  open(PF, ">$pid_file");
  print PF "$oc4j_pid\n";
  close(PF);

  my $read_pid = read_pid_from_oc4j_pid_file($pid_file);
  if ($read_pid == NULL_PID)
  {
    print STDERR " CRS_ERROR:FATAL Error: Could not create $pid_file\n";
    return FAIL_CODE;
  }   
  elsif ($read_pid != $oc4j_pid)
  {
    print STDERR
      " CRS_ERROR:FATAL Error: PID file mismatch (PID $oc4j_pid running," .
      " PID $read_pid read from PID file)\n";
    unlink($pid_file);
    return FAIL_CODE;
  }
  return SUCC_CODE;
}

sub send_signal
#---------------------------------------------------------------------
# Function: Send check, quit or kill signal
#
# Args    : Type of signal to send ("check", "quit" or "kill")
#           Process number
#
# Returns : SUCC_CODE (for "quit" and "kill", operation initiated;
#                      for "check", process still under control)
#           FAIL_CODE (operation not implemented)
#           ERROR_PID (process no longer under control)
#           NULL_PID  (process no longer exists)
#---------------------------------------------------------------------
{
  return s_send_signal(@_);
}

sub perform_jstack
#---------------------------------------------------------------------
# Function: Perform the jstack command 
#
# Args    : Process number
#
# Returns : jstack command return code
#---------------------------------------------------------------------
{
  my ($oc4j_pid) = @_;
  my @jstack_command = (s_get_jstack(), "-l", $oc4j_pid,
			">>",
			catfile($globhash{$j2ee_home_key}, "log",
				"o4j_jstack_stdout.log"));
  return system("@jstack_command");
}

sub get_oc4j_pid
#---------------------------------------------------------------------
# Function: Get the OC4J container PID(s).
#           If more than one, write an error message.
#
# Args    : None
#
# Returns : OC4J Container PID (one OC4J container found)
#           NULL_PID (no OC4J container found)
#           ERROR_PID (multiple OC4J containers found)
#---------------------------------------------------------------------
{
  my $oc4j_pid;

  my @oc4j_pid_list = get_oc4j_pid_list();
  if (@oc4j_pid_list > 1)
  {
    debug_out("... Multiple OC4J Containers are running ...");
    print STDERR
      " CRS_ERROR:FATAL Error: Multiple OC4J Containers running ".
      "with PIDs";
    foreach (@oc4j_pid_list)
    {
      print STDERR " ", $_;
    }
    print STDERR "\n";
    $oc4j_pid = ERROR_PID;
  }
  elsif (@oc4j_pid_list == 1)
  {
    $oc4j_pid = $oc4j_pid_list[0];
  }
  else
  {
    $oc4j_pid = NULL_PID;
  }
  return $oc4j_pid;
}

sub oc4j_start_post_processing
#---------------------------------------------------------------------
# Function: Verify that a single OC4J container is running.
#           If so, write the PID file and check it.
#
# Args    : Pid file
#
# Returns : OC4J Container PID (post-processing successful)
#           NULL_PID (no OC4J container found)
#           ERROR_PID (error condition)
#---------------------------------------------------------------------
{
  my ($pid_file) = @_;
  my $oc4j_pid = get_oc4j_pid();

  if ($oc4j_pid == NULL_PID)
  {
    debug_out("... OC4J Container is not running ...");
  }
  elsif ($oc4j_pid != ERROR_PID)
  {
    debug_out("... OC4J Container running (pid=$oc4j_pid) ...");
    my $rc = write_pid_to_oc4j_pidfile_and_check($pid_file, $oc4j_pid);
    if ($rc != SUCC_CODE)
    {
      # Kill the OC4J process started above and exit
      send_signal("kill", $oc4j_pid);
      $oc4j_pid = ERROR_PID;
    }
  }
  return $oc4j_pid;
}

sub oc4j_check_servlets
#---------------------------------------------------------------------
# Function: Run OC4J check servlets (container and DBWLM)
#           This is a 2 level HTTP listener check
#           1. Check if the OC4J container is up/down
#           2. Check if DBWLM running in the container is up/down.
#
# Args    : None (http port retrieved from environment hash)
#
#
#     OC4J Check Action Modelling
#     Container=UP, DBWLM=UP   => STATE=ONLINE (Resource is o.k.)
#     Container=UP, DBWLM=DOWN => STATE=FAILED (Resource Cleaned/
#                                               Restarted by CRS)
#     Container=DOWN           => STATE=FAILED(Resource Restarted by CRS)

#---------------------------------------------------------------------
{
  # HTTP State Code
  my $url_container;
  my $url_dbwlm;
  my $ua;
  my $req;
  my $response;
  my $req_method = "GET";

  # Read the http port from environment hash
  my $http_port = $envhash{$http_port_key};

  $url_container =
    "http://" . LOCALHOST . ":" . $http_port . "/";
  $url_dbwlm =
    "http://" . LOCALHOST . ":" . $http_port . "/dbwlm-root/dbwlm";

  # create a browser
  $ua = LWP::UserAgent->new();

  # Create request  object for this URL

  # Ping the Container, send request and get response
  $req = HTTP::Request->new($req_method, $url_container);
  $response = $ua->request($req);

  # Check response
  if ($response->is_success)
  {
    # Dump ret from successful container ping
    print STDOUT "$url_container: ", $response, "\n";
    print STDOUT "Retcode (Container): ", $response->code, "\n";
      
    # Now ping DBWLM
    $req = HTTP::Request->new($req_method, $url_dbwlm);
    $response = $ua->request($req);
    if ($response->is_success)
    {
      # Returning resource state=ONLINE to agent
      print STDOUT "$url_dbwlm: ", $response, "\n";
      print STDOUT "Retcode (DBWLM): ", $response->code, "\n";
      return STATE_ONLINE;
    }
    else
    {
      # Returning resource state=FAILED to agent
      print STDERR "$0: Could not fetch $url_dbwlm\n";
      print STDOUT "Retcode (DBWLM): ", $response->code, "\n";
      return STATE_FAILED;
    }
  }
  else
  {
    # Returning resource state=FAILED to agent
    print STDERR "$0: Could not fetch $url_container\n";
    print STDOUT "Retcode (Container): ", $response->code, "\n";
    return STATE_FAILED;
  }
}


sub oc4j_start
#---------------------------------------------------------------------
# Function: Perform OC4J start action
#
# Args    : None
#
# Returns : SUCC_CODE (action succeeded)
#           FAIL_CODE (action failed)
#---------------------------------------------------------------------
{
  print STDOUT "Start OC4J\n";
  my $start_rc = FAIL_CODE;
  my $iter_cnt = 0;

  my $config_dir = $envhash{$config_dir_key};
  my $min_heap_size = $envhash{$min_heap_size_key};
  my $max_heap_size = $envhash{$max_heap_size_key};

  my $jre = $globhash{$jre_key};
  my $j2ee_home = $globhash{$j2ee_home_key};
  my $pid_file = $globhash{$pid_file_key};

  # Collect heap options

  my @jre_heap_options = ("-Xms$min_heap_size", "-Xmx$max_heap_size");
  if (defined($ENV{"JRE_HEAP_OPTIONS"}))
  {
    @jre_heap_options = (split(/\s+/, $ENV{"JRE_HEAP_OPTIONS"}),
                         @jre_heap_options);
  }
  if (defined($envhash{$min_perm_size_key}))
  {
    @jre_heap_options = (@jre_heap_options,
			 "-XX:PermSize=" . $envhash{$min_perm_size_key});
  }
  if (defined($envhash{$max_perm_size_key}))
  {
    @jre_heap_options = (@jre_heap_options,
			 "-XX:MaxPermSize=" . $envhash{$max_perm_size_key});
  }

  # Collect properties

  my @java_property_defs_base
      = ("-Djava.awt.headless=true", "-Ddisable.checkForUpdate=true");

  my @java_path_defs = s_java_path_defs();

  my @java_log_rotate_defs
    = ("-Dstdstream.filesize=100", "-Dstdstream.filenumber=10");

  my @srvm_tracing;
  if (defined($envhash{$srvm_trace_level_key}))
  {
    @srvm_tracing = ("-DTRACING.ENABLED=true",
                     "-DTRACING.LEVEL=". $envhash{$srvm_trace_level_key});
  }
  else
  {
    @srvm_tracing = ("-DTRACING.ENABLED=false");
  }

  my @dbwlm_tracing
    = ("-Doracle.wlm.dbwlmlogger.logging.level=".
       $envhash{$dbwlm_logging_level_key});

  my @dbwlm_test_props;
  if (defined($envhash{$use_scan_ip_key}))
  {
    @dbwlm_test_props = ("-Duse_scan_IP=" . $envhash{$use_scan_ip_key});
  }

  my @rmi_property_defs = ("-Dport.rmi=" . $ENV{"OC4J_PORT"});
  my @java_property_defs = (@java_property_defs_base,
                            @java_path_defs,
                            @java_log_rotate_defs,
                            @srvm_tracing,
                            @dbwlm_tracing,
                            @dbwlm_test_props,
                            @rmi_property_defs);

  my $j2ee_log_dir = catfile($j2ee_home, "log");
  my $j2ee_log_out = catfile($j2ee_log_dir, "oc4j.out");
  my $j2ee_log_err = catfile($j2ee_log_dir, "oc4j.err");
  my $j2ee_stdout_err = catfile($j2ee_log_dir, "oc4j_stdout_err.log");

  my $crs_start_timeout = $ENV{"_CRS_START_TIMEOUT"};
  debug_out("... OC4J start timeout is $crs_start_timeout seconds");
  my $sleep_interval = $envhash{$isleep_interval_key};
  my $start_iter_max = $crs_start_timeout / $sleep_interval;
  my $start_deployment_interval = 15;
  debug_out("... OC4J start iter max is $start_iter_max iterations");
  if ($start_iter_max < 1)
  {
    $sleep_interval = 0;
  }

  # Change directories to $j2ee_log_dir before launching
  chdir($j2ee_log_dir) || 
    print STDERR "Could not change directory to $j2ee_log_dir\n";
  
  # First, check if a previous OC4J process is running
  my $oc4j_pid = get_oc4j_pid();
  if ($oc4j_pid != NULL_PID)
  {
    if ($oc4j_pid != ERROR_PID)
    {
      print STDERR " CRS_ERROR:FATAL Error: OC4J already " .
                   "running with PID ";
      print STDERR "$oc4j_pid\n";
    }
    return $start_rc;
  }

  # Next, remove stale PID file (if any)
  unlink($pid_file);

  # Update the JAZN file
  my $jazn_xml = catfile($config_dir, "system-jazn-data.xml");
  debug_out("... updating $jazn_xml file ...");
  my @command_parms = ("get", "jazn", $jazn_xml);
  my @command = (catfile($ENV{"ORACLE_HOME"}, "bin", "crsctl"),
                 @command_parms);
  debug_out("COMMAND:", @command);
  my $rc = system("@command");
  if ($rc == SUCC_CODE)
  {
    debug_out("... $jazn_xml file update succeeded ...");
  }
  else
  {
    debug_out("... $jazn_xml file update failed ...");
  }

  # Start the container
  @command_parms =  (split(/\s+/, $ENV{"JRE_OPTIONS"}),
                     split(/\s+/, $ENV{"JRE_OPTIONS_SERVER"}),
                     @jre_heap_options, @java_property_defs,
                     "-jar", catfile($j2ee_home, "oc4j.jar"),
                     "-config", catfile($config_dir, "server.xml"),
                     "-out", $j2ee_log_out,
                     "-err", $j2ee_log_err,
                     ">>", $j2ee_stdout_err,
                     "2>&1");

  async_start($jre, @command_parms);

  # Begin OC4J start post processing
  $oc4j_pid = oc4j_start_post_processing($pid_file);
  if ($oc4j_pid == ERROR_PID)
  {
    return $start_rc;
  }

  sleep($start_deployment_interval);

  while ($iter_cnt < $start_iter_max)
  {
    sleep($sleep_interval);

    if ($oc4j_pid == NULL_PID)
    {
      # Continue OC4J start post processing
      $oc4j_pid = oc4j_start_post_processing($pid_file);
      if ($oc4j_pid == ERROR_PID)
      {
        return $start_rc;
      }
    }
    if ($oc4j_pid != NULL_PID)
    {
      # Run OC4J check servlets using HTTP port from environment
      my $rc = oc4j_check_servlets();
      if ($rc == SUCC_CODE)
      {
        $start_rc = SUCC_CODE;
        debug_out("... OC4J Container is ready ...");
        last;
      }
    }
    debug_out("... OC4J Container is not ready " .
              "($iter_cnt/$start_iter_max) ...");
    $iter_cnt++;
  }
  return $start_rc;
}

sub get_oc4j_shutter_pid
#---------------------------------------------------------------------
# Function: Get the OC4J shutter command PID(s).
#           If more than one, write an error message.
#
# Args    : None
#
# Returns : OC4J shutter command PID (one OC4J shutter command found)
#           NULL_PID (no OC4J shutter command found)
#           ERROR_PID (multiple OC4J shutter commands found)
#---------------------------------------------------------------------
{
  my $oc4j_shutter_pid;

  my @oc4j_shutter_pid_list = get_oc4j_shutter_pid_list();
  if (@oc4j_shutter_pid_list > 1)
  {
    debug_out("... Multiple OC4J Shutters are running ...");
    print STDERR
      " CRS_ERROR:FATAL Error: Multiple OC4J Shutters " .
      "running with PIDs";
    foreach (@oc4j_shutter_pid_list)
    {
      print STDERR " ", $_;
    }
    print STDERR "\n";
    $oc4j_shutter_pid = ERROR_PID;
  }
  elsif (@oc4j_shutter_pid_list == 1)
  {
    $oc4j_shutter_pid = $oc4j_shutter_pid_list[0];
  }
  else
  {
    $oc4j_shutter_pid = NULL_PID;
  }
  return $oc4j_shutter_pid;
}

sub oc4j_stop
#---------------------------------------------------------------------
# Function: Perform OC4J stop action
#
# Args    : None
#
# Returns : SUCC_CODE (action succeeded)
#           FAIL_CODE (action failed)
#---------------------------------------------------------------------
{
  print STDOUT "Stop OC4J\n";
  my $stop_rc = FAIL_CODE;
  my $iter_cnt = 0;
  my $rc = SUCC_CODE;

  my $jlib = $globhash{$jlib_key};
  my $jre = $globhash{$jre_key};
  my $j2ee_home = $globhash{$j2ee_home_key};
  my $pid_file = $globhash{$pid_file_key};

  # Collect all properties

  my $crs_stop_timeout = $ENV{"_CRS_STOP_TIMEOUT"};
  debug_out("... OC4J stop timeout is $crs_stop_timeout seconds");
  my $sleep_interval = $envhash{$isleep_interval_key};
  my $stop_iter_max = $crs_stop_timeout / $sleep_interval;
  debug_out("... OC4J stop iter max is $stop_iter_max iterations");
  if ($stop_iter_max < 1)
  {
    $sleep_interval = 0;
  }

  # Change directories to $j2ee_log_dir before launching
  my $j2ee_log_dir = catfile($j2ee_home, "log");
  chdir($j2ee_log_dir) || 
    print STDERR "Could not change directory to $j2ee_log_dir\n";

  # Shut down the container

  my $j2ee_stdout_err = catfile($j2ee_log_dir,
                                "oc4j_shutter_stdout_err.log");

  my $admin_jar = catfile($j2ee_home, "admin.jar");
  my $srvm_jar = catfile($jlib, "srvm.jar");
  my $srvm_has_jar = catfile($jlib, "srvmhas.jar");
  my $consoleapps_jar = catfile($jlib, "consoleapps.jar");
  my $classpath = $admin_jar . $Config{path_sep} .
                  $consoleapps_jar . $Config{path_sep} .
                  $srvm_jar . $Config{path_sep} .
                  $srvm_has_jar;

  my @command_parms = (split(/\s+/, $ENV{"JRE_OPTIONS"}),
                       "-classpath", $classpath,
                       SHUTTER_CLASS,
                       "ormi://" . LOCALHOST . ":" . $ENV{"OC4J_PORT"},
                       ">>", $j2ee_stdout_err,
                       "2>&1");

  async_start($jre, @command_parms);

  my $oc4j_shutter_pid = get_oc4j_shutter_pid();
  if ($oc4j_shutter_pid == ERROR_PID)
  {
    return $stop_rc;
  }
  elsif ($oc4j_shutter_pid != NULL_PID)
  {
    debug_out("... Initial Check - OC4J Shutter JVM waiting " .
              "(pid=$oc4j_shutter_pid) ...");
  }
  else
  {
    # No OC4J Shutter found - This is not expected, but it
    # could be caused by a race condition.  If there is
    # still no OC4J Shutter after wait, assume success.
    debug_out("... Initial Check found no OC4J Shutter ...");
    $oc4j_shutter_pid = ERROR_PID;
  }

  my $oc4j_pid;
  while ($iter_cnt < $stop_iter_max)
  {
    debug_out("... Sleep for $sleep_interval seconds ...");
    sleep($sleep_interval);

    if ($oc4j_shutter_pid != NULL_PID)
    {
      debug_out("... Iteration $iter_cnt Check for OC4J Shutter " .
                "...");
      $oc4j_shutter_pid = get_oc4j_shutter_pid();
      if ($oc4j_shutter_pid == ERROR_PID)
      {
        last;
      }
      elsif ($oc4j_shutter_pid != NULL_PID)
      {
        debug_out("... Iteration $iter_cnt OC4J Shutter ".
                  " waiting " .
                  "(pid=$oc4j_shutter_pid) ...");
      }
    }

    # If there is no more OC4J Shutter JVM, check for OC4J Container
    if ($oc4j_shutter_pid == NULL_PID)
    {
      # First, check if OC4J process is running
      debug_out("... Iteration $iter_cnt Check for OC4J Container ...");
      $oc4j_pid = get_oc4j_pid();
      if ($oc4j_pid == ERROR_PID)
      {
        last;
      }
      elsif ($oc4j_pid == NULL_PID)
      {
        # No OC4J
        # Remove pidfile and say that stop succeeded
        debug_out("... OC4J Container is stopped ...");
        unlink($pid_file);
        last;
      }
      # Normal case - one OC4J
      # If no PIDFILE, OC4J must already be on the way down
      my $read_pid = read_pid_from_oc4j_pid_file($pid_file);
      if ($read_pid == NULL_PID)
      {
        # No OC4J
        # Remove pidfile and say that stop succeeded
        debug_out("... OC4J Container is stopped ...");
        unlink($pid_file);
        $oc4j_pid = NULL_PID;
        last;
      }
      if ($read_pid != $oc4j_pid)
      {
        print STDERR
          " CRS_ERROR:FATAL Error: PID file mismatch (PID " .
          "$oc4j_pid running, PID $read_pid read from PID file)\n";
        $oc4j_pid = ERROR_PID;
        last;
      }
      debug_out("... OC4J Container shutdown pending " .
                "($iter_cnt/$stop_iter_max) ...");
    }
    $iter_cnt++;
  }
  if (($oc4j_shutter_pid == NULL_PID) &&
      ($oc4j_pid == NULL_PID))
  {
    $stop_rc = SUCC_CODE;
  }
  elsif (($oc4j_shutter_pid != NULL_PID) &&
         ($oc4j_shutter_pid != ERROR_PID))
  {
    # Kill the OC4J Shutter
    send_signal("kill", $oc4j_shutter_pid);
    debug_out("... OC4J Shutter timed out, " .
              "killed (pid=$oc4j_shutter_pid) ...");
  }
  elsif (($oc4j_pid != NULL_PID) &&
         ($oc4j_pid != ERROR_PID))
  {
    debug_out("... OC4J Container (pid=$oc4j_pid) was not stopped ...");
  }
  if ($stop_rc != SUCC_CODE)
  {
    print STDOUT "Stop OC4J Failed";
  }
  return $stop_rc;
}

sub oc4j_check
#---------------------------------------------------------------------
# Function: Perform OC4J check action
#
# Args    : None
#
# Returns : STATE_ONLINE  (OC4J is online)
#           STATE_OFFLINE (OC4J is offline)
#           STATE_UNKNOWN (OC4J state is unknown)
#           STATE_FAILED  (OC4J check failed)
#---------------------------------------------------------------------
{
  print STDOUT "Check OC4J\n";
  my $check_rc = STATE_OFFLINE;
  my $pid_file = $globhash{$pid_file_key};

  debug_out("... issuing CHECK action ...");

  # First, check if OC4J process is running
  my $oc4j_pid = NULL_PID;
  my $read_pid = read_pid_from_oc4j_pid_file($pid_file);

  # If PIDFILE contains a non-null PID and it's under our control,
  # accept this as our container
  # If not, investigate matters further
  if (($read_pid != NULL_PID) &&
      (send_signal("check", $read_pid) == SUCC_CODE))
  {
    $oc4j_pid = $read_pid;
  }
  else
  {
    my @oc4j_pid_list = get_oc4j_pid_list();
    if (@oc4j_pid_list == 0)
    {
      # No OC4J
      # Remove pidfile and say that OC4J is OFFLINE
      unlink($pid_file);
      return $check_rc;
    }

    # Next, check for multiple OC4Js
    if (@oc4j_pid_list > 1)
    {
      # Multiple OC4Js
      debug_out("... Multiple OC4J Containers are running ...");
      print STDERR
        " CRS_ERROR:FATAL Error: Multiple OC4J Containers running ".
        "with PIDs";
      foreach (@oc4j_pid_list)
      {
        print STDERR " ", $_;
      }
      print STDERR "\n";

      # Remove pidfile and say that OC4J is UNKNOWN
      unlink($pid_file);
      $check_rc = STATE_UNKNOWN;
      return $check_rc;
    }

    # Normal case - one OC4J
    $oc4j_pid = $oc4j_pid_list[0];
  }
  if ($read_pid == NULL_PID)
  {
    my $rc = write_pid_to_oc4j_pidfile_and_check($pid_file, $oc4j_pid);
    if ($rc != SUCC_CODE)
    {
      # Kill the OC4J process started above
      send_signal("kill", $oc4j_pid);
      # Remove pidfile and say that OC4J is UNKNOWN
      unlink($pid_file);
      $check_rc = STATE_UNKNOWN;
      return $check_rc;
    }
    $read_pid = $oc4j_pid;
  }
  elsif ($read_pid != $oc4j_pid)
  {
    print STDERR
      " CRS_ERROR:FATAL Error: PID file mismatch (PID $oc4j_pid " .
      "running, PID $read_pid read from PID file)\n";
    
    # Remove pidfile and say that OC4J is UNKNOWN
    unlink($pid_file);
    $check_rc = STATE_UNKNOWN;
    return $check_rc;
  }
  debug_out("... Phase 1: OC4J pid check succeeded (pid=$oc4j_pid) ...");

  # Now issue Phase 2 of the check

  $check_rc = oc4j_check_servlets();
  if ($check_rc == STATE_ONLINE)
  {
    debug_out("... Phase 2: OC4J http check succeeded ...");
  }
  else
  {
    debug_out("... Phase 2: OC4J http check failed ...");
  }

  return $check_rc;
}

sub oc4j_clean
#---------------------------------------------------------------------
# Function: Perform OC4J check action
#
# Args    : Number of seconds to wait after issuing SIGQUIT
#
# Returns : SUCC_CODE (action succeeded)
#           FAIL_CODE (action failed)
#---------------------------------------------------------------------
{
  print STDOUT "Clean OC4J\n";
  my $clean_rc = SUCC_CODE;
  my ($sigquit_sleep_interval) = @_;
  my $iter_cnt = 0;

  my $j2ee_home = $globhash{$j2ee_home_key};
  my $pid_file = $globhash{$pid_file_key};

  my $crs_clean_timeout = $ENV{"_CRS_STOP_TIMEOUT"};
  debug_out("... OC4J clean timeout is $crs_clean_timeout seconds");
  my $sleep_interval = $envhash{$isleep_interval_key};
  my $clean_iter_max = $crs_clean_timeout / $sleep_interval;
  debug_out("... OC4J clean iter max is $clean_iter_max iterations");
  if ($clean_iter_max < 1)
  {
    $sleep_interval = 0;
  }

  # Change directories to $j2ee_log_dir before launching
  my $j2ee_log_dir = catfile($j2ee_home, "log");
  chdir($j2ee_log_dir) || 
    print STDERR "Could not change directory to $j2ee_log_dir\n";
  
  my %oc4j_shutter_pid_hash;
  my %oc4j_pid_hash;
  my $more_work = 1;

  while ($iter_cnt < $clean_iter_max)
  {
    if (($iter_cnt > 0) && ($sleep_interval > 0))
    {
      sleep($sleep_interval);
    }
    $more_work = 0;

    $iter_cnt++;

    foreach my $oc4j_shutter_pid (get_oc4j_shutter_pid_list())
    {
      $more_work = 1;
      if (($iter_cnt < $clean_iter_max) &&
          !defined($oc4j_shutter_pid_hash{$oc4j_shutter_pid}))
      {
        # Kill the OC4J Shutter
        send_signal("kill", $oc4j_shutter_pid);
        debug_out("... Cleaned OC4J Shutter JVM shutdown " .
                  "process (pid=$oc4j_shutter_pid) ...");
        $oc4j_shutter_pid_hash{$oc4j_shutter_pid} = 1;
      }
    }

    foreach my $oc4j_pid (get_oc4j_pid_list())
    {
      $more_work = 1;
      if (($iter_cnt < $clean_iter_max) &&
          !defined($oc4j_pid_hash{$oc4j_pid}))
      {
        # Send SIGQUIT to OC4J (to get a dump)
        if(send_signal("quit", $oc4j_pid) == FAIL_CODE)
	{
          # If no SIGQUIT support, perform jstack command instead
          perform_jstack($oc4j_pid);
        }

        # Sleep on request after SIGQUIT or jstack command
        if ($sigquit_sleep_interval >= 0)
        {
          sleep($sigquit_sleep_interval);
        }

        # Send SIGKILL to OC4J
        send_signal("kill", $oc4j_pid);
        debug_out("... Cleaned OC4J Container (pid=$oc4j_pid) ...");
        $oc4j_pid_hash{$oc4j_pid} = 1;
      }
    }
    if (!$more_work)
    {
        last;
    }
  }
  if ($more_work)
  {
    $clean_rc = FAIL_CODE;
    debug_out("... Timed-out cleanup not complete, aborting ...");
  }

  unlink($pid_file);

  return $clean_rc;
}

1;
